<?php
namespace BooklyPro\Lib\Entities;

use Bookly\Lib as BooklyLib;

/**
 * Class GiftCardType
 *
 * @package BooklyPro\Lib\Entities
 */
class GiftCardType extends BooklyLib\Base\Entity
{
    /** @var string */
    protected $title = '';
    /** @var float */
    protected $amount = 0;
    /** @var string */
    protected $start_date;
    /** @var string */
    protected $end_date;
    /** @var int */
    protected $min_appointments = 1;
    /** @var int */
    protected $max_appointments;

    protected static $table = 'bookly_gift_card_types';

    protected static $schema = array(
        'id' => array( 'format' => '%d' ),
        'title' => array( 'format' => '%s' ),
        'amount' => array( 'format' => '%f' ),
        'start_date' => array( 'format' => '%s' ),
        'end_date' => array( 'format' => '%s' ),
        'min_appointments' => array( 'format' => '%d' ),
        'max_appointments' => array( 'format' => '%d' ),
    );

    /**
     * Check if gift has started.
     *
     * @return bool
     */
    public function started()
    {
        if ( $this->start_date ) {
            $today = BooklyLib\Slots\DatePoint::now()->format( 'Y-m-d' );
            if ( $today < $this->start_date ) {
                return false;
            }
        }

        return true;
    }

    /**
     * Check if gift is expired.
     *
     * @return bool
     */
    public function expired()
    {
        if ( $this->end_date ) {
            $today = BooklyLib\Slots\DatePoint::now()->format( 'Y-m-d' );
            if ( $today > $this->end_date ) {
                return true;
            }
        }

        return false;
    }

    /**
     * Check if gift is valid for given cart item.
     *
     * @param BooklyLib\CartItem $cart_item
     * @return bool
     */
    public function validForCartItem( BooklyLib\CartItem $cart_item )
    {
        $gs = new GiftCardTypeService();
        if ( $gs->loadBy( array( 'gift_card_type_id' => $this->id, 'service_id' => $cart_item->getServiceId() ) ) ) {
            $gst = new GiftCardTypeStaff();

            return $gst->loadBy( array( 'gift_card_type_id' => $this->id, 'staff_id' => $cart_item->getStaffId() ) );
        }

        return false;
    }

    /**
     * Check if gift is valid for given cart.
     *
     * @param BooklyLib\Cart $cart
     * @return bool
     */
    public function validForCart( BooklyLib\Cart $cart )
    {
        $valid = false;

        $services = array();
        $cart_items = $cart->getItems();
        foreach ( $cart_items as $item ) {
            if ( $this->validForCartItem( $item ) ) {
                // Count each service in cart.
                $service_id = $item->getServiceId();
                if ( ! isset ( $services[ $service_id ] ) ) {
                    $services[ $service_id ] = 0;
                }
                ++$services[ $service_id ];
            }
        }

        if ( ! empty ( $services ) ) {
            // Find min and max count.
            $min_count = PHP_INT_MAX;
            $max_count = 0;
            foreach ( $services as $count ) {
                if ( $count < $min_count ) {
                    $min_count = $count;
                }
                if ( $count > $max_count ) {
                    $max_count = $count;
                }
            }
            if ( $min_count >= $this->min_appointments ) {
                if ( $this->max_appointments === null || $max_count <= $this->max_appointments ) {
                    $valid = true;
                }
            }
        }

        return $valid;
    }

    /**
     * Get translated title.
     *
     * @param string $locale
     * @return string
     */
    public function getTranslatedTitle( $locale = null )
    {
        return $this->getTitle() != ''
            ? BooklyLib\Utils\Common::getTranslatedString( 'gift_card_type_' . $this->getId(), $this->getTitle(), $locale )
            : '';
    }

    /**************************************************************************
     * Entity Fields Getters & Setters                                        *
     **************************************************************************/

    /**
     * Gets title
     *
     * @return string
     */
    public function getTitle()
    {
        return $this->title;
    }

    /**
     * Sets title
     *
     * @param string $title
     * @return $this
     */
    public function setTitle( $title )
    {
        $this->title = $title;

        return $this;
    }

    /**
     * @return float
     */
    public function getAmount()
    {
        return $this->amount;
    }

    /**
     * @param float $amount
     * @return $this
     */
    public function setAmount( $amount )
    {
        $this->amount = $amount;

        return $this;
    }

    /**
     * Gets start_date
     *
     * @return string
     */
    public function getStartDate()
    {
        return $this->start_date;
    }

    /**
     * Sets start_date
     *
     * @param string $start_date
     * @return $this
     */
    public function setStartDate( $start_date )
    {
        $this->start_date = $start_date;

        return $this;
    }

    /**
     * Gets end_date
     *
     * @return string
     */
    public function getEndDate()
    {
        return $this->end_date;
    }

    /**
     * Sets end_date
     *
     * @param string $end_date
     * @return $this
     */
    public function setEndDate( $end_date )
    {
        $this->end_date = $end_date;

        return $this;
    }

    /**
     * Gets min_appointments
     *
     * @return int
     */
    public function getMinAppointments()
    {
        return $this->min_appointments;
    }

    /**
     * Sets min_appointments
     *
     * @param int $min_appointments
     * @return $this
     */
    public function setMinAppointments( $min_appointments )
    {
        $this->min_appointments = $min_appointments;

        return $this;
    }

    /**
     * Gets max_appointments
     *
     * @return int
     */
    public function getMaxNumberOfAppointments()
    {
        return $this->max_appointments;
    }

    /**
     * Sets max_appointments
     *
     * @param int $max_appointments
     * @return GiftCardType
     */
    public function setMaxNumberOfAppointments( $max_appointments )
    {
        $this->max_appointments = $max_appointments;

        return $this;
    }

    /**************************************************************************
     * Overridden Methods                                                     *
     **************************************************************************/

    /**
     * @inerhitDoc
     */
    public function save()
    {

        $return = parent::save();
        if ( $this->isLoaded() ) {
            // Register string for translate in WPML.
            do_action( 'wpml_register_single_string', 'bookly', 'gift_card_type_' . $this->getId(), $this->getTitle() );
        }

        return $return;
    }
}